<?php
/**
 * User Registration Page
 */
require_once __DIR__ . '/config.php';

if (isLoggedIn()) {
    redirect('/user/dashboard.php');
}

$refCode = sanitize($_GET['ref'] ?? '');
$csrfToken = generateCSRFToken();
$error = '';
$success = '';

// Handle POST
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $csrf = $_POST['csrf_token'] ?? '';
    if (!verifyCSRFToken($csrf)) {
        $error = 'Invalid security token. Please refresh.';
    } else {
        if (!rateLimit('register', 5, 10)) {
            $error = 'Too many attempts. Please wait 10 minutes.';
        } else {
            $name = trim($_POST['name'] ?? '');
            $phone = preg_replace('/[^0-9]/', '', $_POST['phone'] ?? '');
            $password = $_POST['password'] ?? '';
            $confirmPassword = $_POST['confirm_password'] ?? '';
            $referralCode = trim($_POST['referral_code'] ?? '');
            
            // Validate
            if (empty($name) || !validateName($name)) {
                $error = 'Please enter a valid name (2-100 characters, letters only)';
            } elseif (!validatePhone($phone)) {
                $error = 'Please enter a valid 10-digit Indian mobile number';
            } elseif (!validatePassword($password)) {
                $error = 'Password must be 6-50 characters';
            } elseif ($password !== $confirmPassword) {
                $error = 'Passwords do not match';
            } else {
                // Check duplicate
                $existing = getUserByPhone($phone);
                if ($existing) {
                    $error = 'This phone number is already registered. Please login.';
                } else {
                    // Validate referral code if provided
                    if (!empty($referralCode)) {
                        $referrer = getUserByReferralCode($referralCode);
                        if (!$referrer) {
                            $error = 'Invalid referral code';
                        }
                    }
                    
                    if (empty($error)) {
                        try {
                            $userId = createUser($name, $phone, $password, $referralCode ?: null);
                            
                            // Login immediately
                            loginUser($userId);
                            
                            redirect('/user/dashboard.php');
                        } catch (Exception $e) {
                            error_log('Registration Error: ' . $e->getMessage());
                            $error = 'Registration failed. Please try again.';
                        }
                    }
                }
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width,initial-scale=1.0">
    <meta name="theme-color" content="#1a1a2e">
    <title>Register — Hindustan Club</title>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@400;500;600;700&display=swap" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css" rel="stylesheet">
    <link href="/assets/css/style.css" rel="stylesheet">
</head>
<body class="auth-body">
<div class="auth-container">
    <div class="auth-card">
        <!-- Logo -->
        <div class="auth-logo">
            <div class="auth-logo-icon">HC</div>
            <h1 class="auth-title">Create Account</h1>
            <p class="auth-sub">Join Hindustan Club and start winning!</p>
        </div>
        
        <!-- Error -->
        <?php if ($error): ?>
        <div class="auth-alert auth-alert-error">
            <i class="fas fa-exclamation-circle"></i> <?php echo htmlspecialchars($error); ?>
        </div>
        <?php endif; ?>
        
        <!-- Form -->
        <form method="POST" class="auth-form" autocomplete="off" id="registerForm">
            <input type="hidden" name="csrf_token" value="<?php echo $csrfToken; ?>">
            
            <div class="auth-field">
                <label class="auth-label">Full Name</label>
                <div class="auth-input-wrap">
                    <i class="fas fa-user auth-input-icon"></i>
                    <input type="text" name="name" class="auth-input" placeholder="Enter your full name" value="<?php echo htmlspecialchars($_POST['name'] ?? ''); ?>" required maxlength="100" autocomplete="name">
                </div>
            </div>
            
            <div class="auth-field">
                <label class="auth-label">Mobile Number</label>
                <div class="auth-input-wrap">
                    <span class="auth-input-prefix">+91</span>
                    <input type="tel" name="phone" class="auth-input auth-input-phone" placeholder="10-digit mobile number" value="<?php echo htmlspecialchars($_POST['phone'] ?? ''); ?>" required maxlength="10" pattern="[6-9][0-9]{9}" autocomplete="tel">
                </div>
            </div>
            
            <div class="auth-field">
                <label class="auth-label">Password</label>
                <div class="auth-input-wrap">
                    <i class="fas fa-lock auth-input-icon"></i>
                    <input type="password" name="password" id="regPassword" class="auth-input" placeholder="Min 6 characters" required minlength="6" maxlength="50">
                    <button type="button" class="auth-eye-btn" onclick="toggleEye('regPassword', this)"><i class="fas fa-eye"></i></button>
                </div>
            </div>
            
            <div class="auth-field">
                <label class="auth-label">Confirm Password</label>
                <div class="auth-input-wrap">
                    <i class="fas fa-lock auth-input-icon"></i>
                    <input type="password" name="confirm_password" id="regConfirm" class="auth-input" placeholder="Re-enter password" required minlength="6" maxlength="50">
                    <button type="button" class="auth-eye-btn" onclick="toggleEye('regConfirm', this)"><i class="fas fa-eye"></i></button>
                </div>
            </div>
            
            <div class="auth-field">
                <label class="auth-label">Referral Code <span style="color:#666">(optional)</span></label>
                <div class="auth-input-wrap">
                    <i class="fas fa-gift auth-input-icon" style="color:#FFD700"></i>
                    <input type="text" name="referral_code" class="auth-input" placeholder="Enter referral code" value="<?php echo htmlspecialchars($refCode ?: ($_POST['referral_code'] ?? '')); ?>" maxlength="20" style="text-transform:uppercase" autocomplete="off">
                </div>
            </div>
            
            <button type="submit" class="auth-submit-btn" id="registerBtn">
                <i class="fas fa-user-plus"></i> Create Account
            </button>
        </form>
        
        <div class="auth-footer">
            Already have an account? <a href="/login.php" class="auth-link">Login</a>
        </div>
    </div>
</div>

<script>
function toggleEye(inputId, btn){
    var inp=document.getElementById(inputId);
    var icon=btn.querySelector('i');
    if(inp.type==='password'){inp.type='text';icon.className='fas fa-eye-slash';}
    else{inp.type='password';icon.className='fas fa-eye';}
}

document.getElementById('registerForm').addEventListener('submit',function(e){
    var btn=document.getElementById('registerBtn');
    btn.disabled=true;
    btn.innerHTML='<i class="fas fa-spinner fa-spin"></i> Creating Account...';
});
</script>
</body>
</html>