<?php
/**
 * Wallet Helper — 3-Wallet System
 * Include this in config.php: require_once __DIR__ . '/includes/wallet-helper.php';
 *
 * Wallet Types:
 *   deposit  = Real deposited money — used FIRST in games, withdrawable
 *   earning  = Game winnings + referral commissions — withdrawable
 *   bonus    = Promo/gift codes — NOT directly withdrawable, usable in games
 *
 * Game deduction priority: deposit → earning → bonus
 */

function getWallet(int $userId): array
{
    $w = fetch("SELECT * FROM `wallets` WHERE `user_id`=? LIMIT 1", [$userId]);
    if (!$w) {
        insert('wallets', ['user_id'=>$userId,'deposit_balance'=>0,'earning_balance'=>0,'bonus_balance'=>0]);
        return ['deposit_balance'=>0.00,'earning_balance'=>0.00,'bonus_balance'=>0.00,'total_deposited'=>0.00,'total_withdrawn'=>0.00,'total_wagered'=>0.00,'wagering_required'=>0.00];
    }
    return $w;
}

function getTotalBalance(int $userId): float
{
    $w = getWallet($userId);
    return (float)$w['deposit_balance'] + (float)$w['earning_balance'] + (float)$w['bonus_balance'];
}

function getWithdrawableBalance(int $userId): float
{
    $w = getWallet($userId);
    return (float)$w['deposit_balance'] + (float)$w['earning_balance'];
}

function deductFromWallet(int $userId, float $amount, string $txType = 'bet', string $desc = ''): array
{
    $w = getWallet($userId);
    $remaining = $amount;
    $d = ['deposit'=>0.0,'earning'=>0.0,'bonus'=>0.0];

    if ($remaining > 0 && $w['deposit_balance'] > 0) {
        $take = min($remaining, (float)$w['deposit_balance']);
        $d['deposit'] = $take; $remaining -= $take;
    }
    if ($remaining > 0 && $w['earning_balance'] > 0) {
        $take = min($remaining, (float)$w['earning_balance']);
        $d['earning'] = $take; $remaining -= $take;
    }
    if ($remaining > 0 && $w['bonus_balance'] > 0) {
        $take = min($remaining, (float)$w['bonus_balance']);
        $d['bonus'] = $take; $remaining -= $take;
    }
    if ($remaining > 0.005) throw new \RuntimeException('Insufficient balance');

    update('wallets', [
        'deposit_balance' => max(0, round((float)$w['deposit_balance'] - $d['deposit'], 2)),
        'earning_balance' => max(0, round((float)$w['earning_balance'] - $d['earning'], 2)),
        'bonus_balance'   => max(0, round((float)$w['bonus_balance'] - $d['bonus'], 2)),
        'total_wagered'   => round((float)$w['total_wagered'] + $amount, 2),
    ], 'user_id=?', [$userId]);

    $primary = $d['deposit']>0 ? 'deposit' : ($d['earning']>0 ? 'earning' : 'bonus');
    insert('transactions', ['user_id'=>$userId,'type'=>$txType,'amount'=>-$amount,'wallet_type'=>$primary,'balance_after'=>getTotalBalance($userId),'status'=>'completed','description'=>$desc]);
    return $d;
}

function creditToWallet(int $userId, float $amount, string $walletType = 'earning', string $txType = 'win', string $desc = ''): void
{
    if (!in_array($walletType, ['deposit','earning','bonus'])) $walletType = 'earning';
    $col = $walletType.'_balance';
    $w = getWallet($userId);
    $extra = $walletType === 'deposit' ? ['total_deposited'=>round((float)$w['total_deposited']+$amount,2)] : [];
    update('wallets', array_merge([$col=>round((float)$w[$col]+$amount,2)], $extra), 'user_id=?', [$userId]);
    insert('transactions', ['user_id'=>$userId,'type'=>$txType,'amount'=>$amount,'wallet_type'=>$walletType,'balance_after'=>getTotalBalance($userId),'status'=>'completed','description'=>$desc]);
}

function creditDeposit(int $userId, float $amount, float $bonus = 0): void
{
    creditToWallet($userId, $amount, 'deposit', 'deposit', 'Deposit approved');
    if ($bonus > 0) creditToWallet($userId, $bonus, 'bonus', 'deposit_bonus', 'First deposit bonus');
}

function creditGameWin(int $userId, float $amount, string $game = ''): void
{
    creditToWallet($userId, $amount, 'earning', 'game_win', "Won: $game");
}

function creditReferralCommission(int $userId, float $amount, string $desc = 'Referral commission'): void
{
    creditToWallet($userId, $amount, 'earning', 'referral', $desc);
}

function creditBonus(int $userId, float $amount, string $desc = 'Bonus'): void
{
    creditToWallet($userId, $amount, 'bonus', 'bonus', $desc);
}

function deductForWithdrawal(int $userId, float $amount): void
{
    $w = getWallet($userId);
    $avail = (float)$w['deposit_balance'] + (float)$w['earning_balance'];
    if ($avail < $amount) throw new \RuntimeException('Insufficient withdrawable balance. Available: ₹'.number_format($avail,2));
    $rem = $amount; $nd = (float)$w['deposit_balance']; $ne = (float)$w['earning_balance'];
    if ($rem > 0 && $nd > 0) { $t=min($rem,$nd); $nd-=$t; $rem-=$t; }
    if ($rem > 0 && $ne > 0) { $t=min($rem,$ne); $ne-=$t; $rem-=$t; }
    update('wallets', ['deposit_balance'=>max(0,round($nd,2)),'earning_balance'=>max(0,round($ne,2)),'total_withdrawn'=>round((float)$w['total_withdrawn']+$amount,2)], 'user_id=?', [$userId]);
    insert('transactions', ['user_id'=>$userId,'type'=>'withdrawal','amount'=>-$amount,'wallet_type'=>'deposit','balance_after'=>getTotalBalance($userId),'status'=>'completed','description'=>'Withdrawal']);
}

function getActiveUpiForDeposit(): ?array
{
    $mode = getSetting('upi_collection_mode', 'multi');
    if ($mode === 'single') {
        return fetch("SELECT * FROM `admin_upi_settings` WHERE `is_active`=1 ORDER BY `priority` DESC LIMIT 1");
    }
    $upis = fetchAll("SELECT * FROM `admin_upi_settings` WHERE `is_active`=1 ORDER BY `priority` DESC");
    foreach ($upis as $u) {
        if ((float)$u['today_received'] < (float)$u['daily_limit']) return $u;
    }
    return $upis[0] ?? null;
}
