<?php
/**
 * Withdrawal API
 */
require_once __DIR__ . '/../config.php';
requireLogin();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') jsonResponse(false, 'Invalid method');
$csrf = $_POST['csrf_token'] ?? '';
if (!verifyCSRFToken($csrf)) jsonResponse(false, 'Invalid token');

$userId = getCurrentUserId();
$action = sanitize($_POST['action'] ?? '');

if ($action !== 'submit_withdrawal') jsonResponse(false, 'Invalid action');

if (!rateLimit('withdrawal_' . $userId, 3, 5)) {
    jsonResponse(false, 'Too many requests. Please wait.');
}

$user = getCurrentUser();
$wallet = getWallet($userId);
$amount = round(floatval($_POST['amount'] ?? 0), 2);
$paymentMethod = sanitize($_POST['payment_method'] ?? '');
$pin = $_POST['pin'] ?? '';

$minWd        = (float)getSetting('min_withdrawal', '200');
$withdrawable = getWithdrawableBalance($userId); // deposit + earning only (bonus exclude)
$dailyLimit   = getDailyWithdrawalLimit($userId);
$todayWithdrawn = getTodayWithdrawn($userId);
$remaining    = $dailyLimit - $todayWithdrawn;

// Validate PIN
if (!empty($user['transaction_pin'])) {
    if (!password_verify($pin, $user['transaction_pin'])) {
        jsonResponse(false, 'Incorrect Transaction PIN');
    }
} else {
    jsonResponse(false, 'Please set a transaction PIN first');
}

// Validate amount
if ($amount < $minWd)        jsonResponse(false, 'Minimum withdrawal is ₹' . number_format($minWd));
if ($amount > $withdrawable) jsonResponse(false, 'Insufficient balance. Withdrawable: ₹' . number_format($withdrawable, 2) . ' (Bonus wallet cannot be withdrawn directly)');
if ($amount > $remaining)    jsonResponse(false, 'Daily withdrawal limit exceeded. Remaining: ₹' . number_format($remaining));

// Wagering check
$wagered = (float)$wallet['total_wagered'];
$required = (float)$wallet['wagering_required'];
if ($wagered < $required) {
    jsonResponse(false, 'Wagering requirement not met. Wagered: ₹' . number_format($wagered) . ' / Required: ₹' . number_format($required));
}

// Validate payment method
if (!in_array($paymentMethod, ['upi', 'bank'])) {
    jsonResponse(false, 'Invalid payment method');
}

// Build withdrawal data (3-wallet system: sirf deposit+earning se katega)
$wdData = ['user_id'=>$userId, 'amount'=>$amount, 'method'=>$paymentMethod, 'status'=>'pending'];

if ($paymentMethod === 'upi') {
    $upiId = sanitize(trim($_POST['upi_id'] ?? ''));
    if (!validateUPI($upiId)) jsonResponse(false, 'Invalid UPI ID');
    $bd = fetch("SELECT * FROM `user_bank_details` WHERE `user_id`=? AND `type`='upi' AND `upi_id`=? LIMIT 1", [$userId, $upiId]);
    $wdData['bank_detail_id'] = $bd['id'] ?? null;
} else {
    $bdId = intval($_POST['bank_detail_id'] ?? 0);
    $bd = fetch("SELECT * FROM `user_bank_details` WHERE `id`=? AND `user_id`=? AND `type`='bank' LIMIT 1", [$bdId, $userId]);
    if (!$bd) jsonResponse(false, 'Invalid bank account');
    $wdData['bank_detail_id'] = $bdId;
}

try {
    $pdo = db();
    $pdo->beginTransaction();
    
    // 3-wallet: deposit -> earning se katao (bonus excluded)
    deductForWithdrawal($userId, $amount);
    query("UPDATE `users` SET `total_withdrawn`=`total_withdrawn`+? WHERE `id`=?", [$amount, $userId]);
    
    insert('withdrawals', $wdData);
    addNotification($userId, '📤 Withdrawal Submitted', 'Your withdrawal of ₹' . number_format($amount) . ' is being processed. Time: ' . getSetting('withdrawal_processing_time', '1-24 hours'), 'withdrawal');
    
    $pdo->commit();
    jsonResponse(true, 'Withdrawal submitted! Processing in ' . getSetting('withdrawal_processing_time', '1-24 hours') . '.');
    
} catch (Exception $e) {
    if (db()->inTransaction()) db()->rollBack();
    error_log('Withdrawal Error: ' . $e->getMessage());
    jsonResponse(false, 'Failed: ' . $e->getMessage());
}