<?php
/**
 * Wingo Game API
 */
require_once __DIR__ . '/../config.php';
requireLogin();

$userId = getCurrentUserId();

// GET requests
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    $action = sanitize($_GET['action'] ?? '');
    
    if ($action === 'get_result') {
        $periodId = intval($_GET['period_id'] ?? 0);
        $period = fetch("SELECT * FROM `wingo_periods` WHERE `id` = ? LIMIT 1", [$periodId]);
        
        if (!$period || $period['status'] !== 'resulted') {
            jsonResponse(false, 'Result not ready');
        }
        
        // Get user's bets for this period
        $myResults = fetchAll(
            "SELECT * FROM `wingo_bets` WHERE `user_id` = ? AND `period_id` = ?",
            [$userId, $periodId]
        );
        
        jsonResponse(true, '', [
            'result_number' => (int)$period['result_number'],
            'result_color'  => $period['result_color'],
            'result_size'   => $period['result_size'],
            'my_results'    => $myResults,
        ]);
    }
    
    if ($action === 'get_period') {
        $duration = intval($_GET['duration'] ?? 1);
        $period = getCurrentWingoPeriod($duration);
        jsonResponse(true, '', ['period' => $period]);
    }
    
    jsonResponse(false, 'Invalid action');
}

// POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') jsonResponse(false, 'Invalid method');
$csrf = $_POST['csrf_token'] ?? '';
if (!verifyCSRFToken($csrf)) jsonResponse(false, 'Invalid token');

$action = sanitize($_POST['action'] ?? '');

if ($action === 'place_bet') {
    if (!rateLimit('wingo_bet_' . $userId, 20, 1)) {
        jsonResponse(false, 'Too many bets. Slow down.');
    }
    
    $periodId = intval($_POST['period_id'] ?? 0);
    $periodNumber = sanitize($_POST['period_number'] ?? '');
    $gameDuration = intval($_POST['game_duration'] ?? 0);
    $betType = sanitize($_POST['bet_type'] ?? '');
    $betValue = sanitize(strtolower($_POST['bet_value'] ?? ''));
    $betAmount = round(floatval($_POST['bet_amount'] ?? 0), 2);
    $multiplier = intval($_POST['multiplier'] ?? 1);
    
    // Validate duration
    if (!in_array($gameDuration, [1, 3, 5, 10])) {
        jsonResponse(false, 'Invalid game duration');
    }
    
    // Validate bet type
    if (!in_array($betType, ['color', 'number', 'size'])) {
        jsonResponse(false, 'Invalid bet type');
    }
    
    // Validate bet value
    $validValues = [
        'color'  => ['green', 'red', 'violet'],
        'number' => ['0','1','2','3','4','5','6','7','8','9'],
        'size'   => ['small', 'big'],
    ];
    if (!in_array($betValue, $validValues[$betType])) {
        jsonResponse(false, 'Invalid bet value');
    }
    
    // Validate amount
    if ($betAmount < 10) jsonResponse(false, 'Minimum bet is ₹10');
    if ($betAmount > 50000) jsonResponse(false, 'Maximum bet is ₹50,000');
    if (!in_array($multiplier, [1, 2, 5, 10])) $multiplier = 1;
    
    $totalAmount = round($betAmount * $multiplier, 2);
    if ($totalAmount > 50000) jsonResponse(false, 'Total bet cannot exceed ₹50,000');
    
    // Validate period is still open
    $period = fetch("SELECT * FROM `wingo_periods` WHERE `id` = ? AND `status` = 'betting' AND `end_time` > DATE_ADD(NOW(), INTERVAL 5 SECOND) LIMIT 1", [$periodId]);
    if (!$period) {
        jsonResponse(false, 'Betting is closed for this period');
    }
    
    // Check balance
    $totalBalance = getTotalBalance($userId);
    if ($totalBalance < $totalAmount) {
        jsonResponse(false, 'Insufficient balance. Available: ₹' . number_format($totalBalance, 2));
    }
    
    // Calculate potential win
    $potentialWin = 0;
    if ($betType === 'number') $potentialWin = $totalAmount * 9;
    elseif ($betType === 'color' && $betValue === 'violet') $potentialWin = $totalAmount * 4.5;
    else $potentialWin = $totalAmount * 2;
    
    try {
        $pdo = db();
        $pdo->beginTransaction();
        
        // Deduct balance (main first, then bonus)
        deductFromWallet($userId, $totalAmount, 'bet', "Wingo {$gameDuration}m #{$periodNumber} — {$betType}:{$betValue}");
        
        // Insert bet
        $betId = insert('wingo_bets', [
            'user_id'        => $userId,
            'period_id'      => $periodId,
            'period_number'  => $periodNumber,
            'game_duration'  => $gameDuration,
            'bet_type'       => $betType,
            'bet_value'      => $betValue,
            'bet_amount'     => $betAmount,
            'multiplier'     => $multiplier,
            'total_amount'   => $totalAmount,
            'potential_win'  => round($potentialWin, 2),
            'result'         => 'pending',
            'win_amount'     => 0,
        ]);
        
        // Update period total
        query("UPDATE `wingo_periods` SET `total_bet_amount` = `total_bet_amount` + ? WHERE `id` = ?", [$totalAmount, $periodId]);
        
        // Process referral commission
        processReferralCommission($userId, $totalAmount, 'Wingo');
        
        $pdo->commit();
        
        jsonResponse(true, 'Bet placed successfully!', [
            'bet_id'  => $betId,
            'total'   => $totalAmount,
            'balance' => getTotalBalance($userId),
        ]);
        
    } catch (Exception $e) {
        if (db()->inTransaction()) db()->rollBack();
        error_log('Wingo Bet Error: ' . $e->getMessage());
        jsonResponse(false, $e->getMessage());
    }
}

jsonResponse(false, 'Invalid action');