<?php
/**
 * Spin & Win API
 */
require_once __DIR__ . '/../config.php';
requireLogin();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') jsonResponse(false, 'Invalid method');
$csrf = $_POST['csrf_token'] ?? '';
if (!verifyCSRFToken($csrf)) jsonResponse(false, 'Invalid token');

$userId = getCurrentUserId();
$action = sanitize($_POST['action'] ?? '');

if ($action !== 'spin') jsonResponse(false, 'Invalid action');

if (!rateLimit('spin_' . $userId, 10, 1)) {
    jsonResponse(false, 'Too fast! Wait a moment.');
}

$spinType = sanitize($_POST['spin_type'] ?? '');
if (!in_array($spinType, ['free', 'paid'])) {
    jsonResponse(false, 'Invalid spin type');
}

$spinCost = (float)getSetting('spin_cost', '10');
$freeDailySpins = (int)getSetting('free_spins_daily', '1');

// Check free spins
if ($spinType === 'free') {
    $todayFree = (int)(fetch("SELECT COUNT(*) as c FROM `spin_history` WHERE `user_id`=? AND `spin_type`='free' AND DATE(`created_at`)=CURDATE()", [$userId])['c'] ?? 0);
    if ($todayFree >= $freeDailySpins) {
        jsonResponse(false, 'No free spins left today! Try paid spin.');
    }
}

// Check balance for paid
if ($spinType === 'paid') {
    $balance = getTotalBalance($userId);
    if ($balance < $spinCost) {
        jsonResponse(false, 'Insufficient balance. Need ₹' . number_format($spinCost));
    }
}

// Get active segments
$segments = fetchAll("SELECT * FROM `spin_config` WHERE `is_active`=1 ORDER BY `segment_number` ASC");
if (empty($segments)) jsonResponse(false, 'Spin not configured');

// Determine winning segment using weighted probability
$totalProb = 0;
foreach ($segments as $s) $totalProb += (float)$s['probability'];
$rand = mt_rand(0, (int)($totalProb * 100)) / 100;
$cumulative = 0;
$winSegment = $segments[0]; // fallback
$winSegmentIndex = 0;

foreach ($segments as $idx => $seg) {
    $cumulative += (float)$seg['probability'];
    if ($rand <= $cumulative) {
        $winSegment = $seg;
        $winSegmentIndex = $idx;
        break;
    }
}

$winAmount = (float)$winSegment['amount'];
$cost = $spinType === 'paid' ? $spinCost : 0;

try {
    $pdo = db();
    $pdo->beginTransaction();
    
    // Deduct cost for paid spin
    if ($spinType === 'paid') {
        deductFromWallet($userId, $spinCost, 'bet', 'Spin & Win — paid spin');
        processReferralCommission($userId, $spinCost, 'Spin');
    }
    
    // Add winnings
    if ($winAmount > 0) {
        addToBonusWallet($userId, $winAmount, 'spin_reward', 'Spin & Win — won ' . $winSegment['label']);
    }
    
    // Record spin
    insert('spin_history', [
        'user_id'     => $userId,
        'spin_type'   => $spinType,
        'cost'        => $cost,
        'segment_won' => (int)$winSegment['segment_number'],
        'amount_won'  => $winAmount,
    ]);
    
    // Track for algorithm
    if ($spinType === 'paid') {
        trackUserBet($userId, $spinCost, true, $winAmount);
    }
    
    // Notification
    addNotification($userId, '🎡 Spin Winner!', 'You won ' . $winSegment['label'] . '! +₹' . number_format($winAmount, 2), 'game');
    
    $pdo->commit();
    
    jsonResponse(true, 'You won ' . $winSegment['label'] . '!', [
        'segment_won'   => $winSegmentIndex,
        'segment_number'=> (int)$winSegment['segment_number'],
        'label'         => $winSegment['label'],
        'amount_won'    => $winAmount,
        'balance'       => getTotalBalance($userId),
    ]);
    
} catch (Exception $e) {
    if (db()->inTransaction()) db()->rollBack();
    error_log('Spin Error: ' . $e->getMessage());
    jsonResponse(false, $e->getMessage());
}