<?php
/**
 * Gift Code Redemption API
 */
require_once __DIR__ . '/../config.php';
requireLogin();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') jsonResponse(false, 'Invalid method');
$csrf = $_POST['csrf_token'] ?? '';
if (!verifyCSRFToken($csrf)) jsonResponse(false, 'Invalid token');

$userId = getCurrentUserId();
$action = sanitize($_POST['action'] ?? '');

if ($action !== 'redeem') jsonResponse(false, 'Invalid action');

if (!rateLimit('giftcode_' . $userId, 5, 5)) {
    jsonResponse(false, 'Too many attempts. Wait 5 minutes.');
}

$code = strtoupper(trim(sanitize($_POST['code'] ?? '')));
if (empty($code) || strlen($code) < 3 || strlen($code) > 30) {
    jsonResponse(false, 'Invalid code format');
}

// Find code
$gift = fetch("SELECT * FROM `gift_codes` WHERE `code` = ? AND `is_active` = 1 LIMIT 1", [$code]);
if (!$gift) {
    jsonResponse(false, 'Invalid or expired gift code');
}

// Check expiry
if (!empty($gift['expires_at']) && strtotime($gift['expires_at']) < time()) {
    jsonResponse(false, 'This gift code has expired');
}

// Check max uses
if ((int)$gift['max_uses'] > 0 && (int)$gift['used_count'] >= (int)$gift['max_uses']) {
    jsonResponse(false, 'This gift code has reached its maximum uses');
}

// Check per-user limit
$userUsage = (int)(fetch("SELECT COUNT(*) as c FROM `gift_code_usage` WHERE `gift_code_id` = ? AND `user_id` = ?", [$gift['id'], $userId])['c'] ?? 0);
if ($userUsage >= (int)$gift['max_per_user']) {
    jsonResponse(false, 'You have already used this code');
}

// Check minimum deposit requirement
if ((float)$gift['min_deposit'] > 0) {
    $user = getCurrentUser();
    if ((float)$user['total_deposited'] < (float)$gift['min_deposit']) {
        jsonResponse(false, 'Minimum deposit of ₹' . number_format((float)$gift['min_deposit']) . ' required to use this code');
    }
}

$amount = (float)$gift['bonus_amount'];

try {
    $pdo = db();
    $pdo->beginTransaction();
    
    // Add to bonus wallet
    addToBonusWallet($userId, $amount, 'gift_code', 'Gift code: ' . $code);
    
    // Add wagering requirement
    $wageringMult = (float)($gift['wagering_multiplier'] ?? 5);
    $wagering = $amount * $wageringMult;
    query("UPDATE `wallets` SET `wagering_required` = `wagering_required` + ? WHERE `user_id` = ?", [$wagering, $userId]);
    
    // Record usage
    insert('gift_code_usage', [
        'gift_code_id' => $gift['id'],
        'user_id'      => $userId,
        'amount'        => $amount,
    ]);
    
    // Increment used count
    query("UPDATE `gift_codes` SET `used_count` = `used_count` + 1 WHERE `id` = ?", [$gift['id']]);
    
    // Notification
    addNotification($userId, '🎟️ Gift Code Redeemed!', 'Code ' . $code . ': +₹' . number_format($amount, 2) . ' bonus added!', 'bonus');
    
    $pdo->commit();
    
    jsonResponse(true, '₹' . number_format($amount, 2) . ' bonus added to your wallet!', [
        'amount'  => $amount,
        'balance' => getTotalBalance($userId),
    ]);
    
} catch (Exception $e) {
    if (db()->inTransaction()) db()->rollBack();
    error_log('Gift Code Error: ' . $e->getMessage());
    jsonResponse(false, 'Failed to redeem. Please try again.');
}