<?php
/**
 * Deposit API
 */
require_once __DIR__ . '/../config.php';
requireLogin();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') jsonResponse(false, 'Invalid method');
$csrf = $_POST['csrf_token'] ?? '';
if (!verifyCSRFToken($csrf)) jsonResponse(false, 'Invalid token');

$userId = getCurrentUserId();
$action = sanitize($_POST['action'] ?? '');

if ($action !== 'submit_deposit') jsonResponse(false, 'Invalid action');

if (!rateLimit('deposit_' . $userId, 5, 5)) {
    jsonResponse(false, 'Too many requests. Please wait 5 minutes.');
}

$amount = round(floatval($_POST['amount'] ?? 0), 2);
$utr = sanitize(trim($_POST['utr_number'] ?? ''));
$upiUsed = sanitize(trim($_POST['upi_id_used'] ?? ''));

$minDeposit = (float)getSetting('min_deposit', '100');
$maxDeposit = (float)getSetting('max_deposit', '50000');

// Validate
if ($amount < $minDeposit) jsonResponse(false, 'Minimum deposit is ₹' . number_format($minDeposit));
if ($amount > $maxDeposit) jsonResponse(false, 'Maximum deposit is ₹' . number_format($maxDeposit));
if (strlen($utr) < 6 || strlen($utr) > 30) jsonResponse(false, 'Invalid UTR number');
if (empty($upiUsed)) jsonResponse(false, 'Please select a UPI ID');

// Check duplicate UTR
$dupUTR = fetch("SELECT * FROM `deposits` WHERE `utr_number` = ? AND `status` != 'rejected' LIMIT 1", [$utr]);
if ($dupUTR) jsonResponse(false, 'This UTR number has already been submitted');

// Check pending limit
$pendingCount = (int)(fetch("SELECT COUNT(*) as c FROM `deposits` WHERE `user_id`=? AND `status`='pending'", [$userId])['c'] ?? 0);
if ($pendingCount >= 5) jsonResponse(false, 'You have too many pending deposits. Please wait for approval.');

// Handle screenshot upload
$screenshotName = null;
if (!empty($_FILES['screenshot']) && $_FILES['screenshot']['error'] === UPLOAD_ERR_OK) {
    $file = $_FILES['screenshot'];
    $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    
    if (!in_array($ext, ['jpg', 'jpeg', 'png', 'webp'])) {
        jsonResponse(false, 'Invalid screenshot format (jpg, png, webp only)');
    }
    if ($file['size'] > 5 * 1024 * 1024) {
        jsonResponse(false, 'Screenshot too large (max 5MB)');
    }
    
    $screenshotName = 'dep_' . $userId . '_' . time() . '_' . bin2hex(random_bytes(4)) . '.' . $ext;
    $uploadPath = UPLOADS_DIR . '/deposits/' . $screenshotName;
    
    if (!move_uploaded_file($file['tmp_name'], $uploadPath)) {
        $screenshotName = null;
    }
}

// Calculate bonus
$user = getCurrentUser();
$isFirstDeposit = empty($user['first_deposit_done']) || (int)$user['first_deposit_done'] === 0;
$bonusAmount = 0;
if ($isFirstDeposit) {
    $realBonusPct = (float)getSetting('first_deposit_real_bonus', '2');
    $bonusAmount = round($amount * $realBonusPct / 100, 2);
}

try {
    insert('deposits', [
        'user_id'      => $userId,
        'amount'       => $amount,
        'upi_id_used'  => $upiUsed,
        'utr_number'   => $utr,
        'screenshot'   => $screenshotName ? 'deposits/' . $screenshotName : null,
        'bonus_amount' => $bonusAmount,
        'status'       => 'pending',
    ]);
    
    addNotification($userId, '📥 Deposit Submitted', 'Your deposit of ₹' . number_format($amount) . ' is under review. UTR: ' . $utr, 'deposit');
    
    jsonResponse(true, 'Deposit submitted successfully! We will review it shortly.');
    
} catch (Exception $e) {
    error_log('Deposit Error: ' . $e->getMessage());
    jsonResponse(false, 'Failed to submit deposit. Please try again.');
}