<?php
/**
 * Daily Check-in API
 */
require_once __DIR__ . '/../config.php';
requireLogin();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') jsonResponse(false, 'Invalid method');
$csrf = $_POST['csrf_token'] ?? '';
if (!verifyCSRFToken($csrf)) jsonResponse(false, 'Invalid security token');

$userId = getCurrentUserId();
$action = sanitize($_POST['action'] ?? '');

if ($action !== 'claim') jsonResponse(false, 'Invalid action');

if (!rateLimit('checkin_' . $userId, 3, 1)) {
    jsonResponse(false, 'Too many requests');
}

// Check if already claimed today
$todayCheckin = fetch("SELECT * FROM `daily_checkin` WHERE `user_id` = ? AND `checkin_date` = CURDATE()", [$userId]);
if ($todayCheckin) {
    jsonResponse(false, 'Already claimed today! Come back tomorrow.');
}

// Calculate current day in streak
$recentCheckins = fetchAll("SELECT * FROM `daily_checkin` WHERE `user_id` = ? ORDER BY `checkin_date` DESC LIMIT 7", [$userId]);

$currentDay = 1;
if (!empty($recentCheckins)) {
    $lastCheckin = $recentCheckins[0];
    $lastDate = $lastCheckin['checkin_date'];
    $yesterday = date('Y-m-d', strtotime('-1 day'));
    
    if ($lastDate === $yesterday) {
        $currentDay = ((int)$lastCheckin['day_number'] % 7) + 1;
    } else {
        $currentDay = 1; // Streak broken
    }
}

// Get reward amount
$config = fetch("SELECT `amount` FROM `daily_checkin_config` WHERE `day_number` = ? AND `is_active` = 1", [$currentDay]);
$amount = $config ? (float)$config['amount'] : 2.00;

try {
    $pdo = db();
    $pdo->beginTransaction();
    
    // Record checkin
    insert('daily_checkin', [
        'user_id'      => $userId,
        'checkin_date'  => date('Y-m-d'),
        'day_number'    => $currentDay,
        'amount'        => $amount,
    ]);
    
    // Add to bonus wallet
    addToBonusWallet($userId, $amount, 'checkin_bonus', 'Day ' . $currentDay . ' check-in bonus');
    
    // Add wagering requirement
    $wageringMult = (float)getSetting('bonus_wagering_multiplier', '50');
    $wagering = $amount * $wageringMult;
    query("UPDATE `wallets` SET `wagering_required` = `wagering_required` + ? WHERE `user_id` = ?", [$wagering, $userId]);
    
    // Notification
    addNotification($userId, '📅 Check-in Bonus!', 'Day ' . $currentDay . ' bonus: +₹' . number_format($amount, 2) . '!', 'bonus');
    
    $pdo->commit();
    
    jsonResponse(true, 'Day ' . $currentDay . ' bonus claimed!', [
        'amount'     => $amount,
        'day_number' => $currentDay,
        'balance'    => getTotalBalance($userId),
    ]);
    
} catch (Exception $e) {
    if (db()->inTransaction()) db()->rollBack();
    error_log('Checkin Error: ' . $e->getMessage());
    jsonResponse(false, 'Failed to claim. Try again.');
}