<?php
/**
 * Admin Settings API (v2 - UPI Multi + Logo/Favicon)
 */
require_once __DIR__ . '/../includes/admin-check.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') jsonResponse(false, 'Invalid method');
$csrf = $_POST['csrf_token'] ?? '';
if (!verifyCSRFToken($csrf)) jsonResponse(false, 'Invalid token');

$action = sanitize($_POST['action'] ?? '');

switch ($action) {
    case 'update_settings':
        $settings = $_POST['settings'] ?? [];
        if (!is_array($settings)) jsonResponse(false, 'Invalid data');
        foreach ($settings as $key => $value) {
            $key = sanitize($key);
            $value = sanitize($value);
            $existing = fetch("SELECT * FROM `site_settings` WHERE `key`=?", [$key]);
            if ($existing) {
                update('site_settings', ['value'=>$value], '`key`=?', [$key]);
            } else {
                insert('site_settings', ['key'=>$key, 'value'=>$value]);
            }
        }
        logAdmin('update_settings', 'Updated site settings: '.implode(', ', array_keys($settings)));
        jsonResponse(true, 'Settings saved');
        break;

    case 'add_upi':
        $upiId = sanitize(trim($_POST['upi_id'] ?? ''));
        $displayName = sanitize(trim($_POST['display_name'] ?? 'Hindustan Club'));
        $minAmount = floatval($_POST['min_amount'] ?? 100);
        $maxAmount = floatval($_POST['max_amount'] ?? 50000);
        $dailyLimit = floatval($_POST['daily_limit'] ?? 500000);
        $priority = intval($_POST['priority'] ?? 0);
        if (empty($upiId)) jsonResponse(false, 'UPI ID required');

        // Handle QR upload
        $qrImage = null;
        if (!empty($_FILES['qr_file']) && $_FILES['qr_file']['error'] === UPLOAD_ERR_OK) {
            $qrImage = handleQrUpload($_FILES['qr_file']);
        }

        insert('admin_upi_settings', [
            'upi_id'        => $upiId,
            'display_name'  => $displayName,
            'qr_code_image' => $qrImage,
            'is_active'     => 1,
            'priority'      => $priority,
            'min_amount'    => $minAmount,
            'max_amount'    => $maxAmount,
            'daily_limit'   => $dailyLimit,
        ]);
        logAdmin('add_upi', "Added UPI: $upiId");
        jsonResponse(true, 'UPI added');
        break;

    case 'edit_upi':
        $id = intval($_POST['upi_id_record'] ?? 0);
        if ($id <= 0) jsonResponse(false, 'Invalid ID');
        update('admin_upi_settings', [
            'upi_id'       => sanitize(trim($_POST['upi_id'] ?? '')),
            'display_name' => sanitize(trim($_POST['display_name'] ?? '')),
            'min_amount'   => floatval($_POST['min_amount'] ?? 100),
            'max_amount'   => floatval($_POST['max_amount'] ?? 50000),
            'daily_limit'  => floatval($_POST['daily_limit'] ?? 500000),
            'priority'     => intval($_POST['priority'] ?? 0),
        ], 'id=?', [$id]);
        logAdmin('edit_upi', "Edited UPI #$id");
        jsonResponse(true, 'UPI updated');
        break;

    case 'toggle_upi':
        $id = intval($_POST['upi_id_record'] ?? 0);
        $active = intval($_POST['is_active'] ?? 0);
        update('admin_upi_settings', ['is_active'=>$active], 'id=?', [$id]);
        logAdmin('toggle_upi', "Toggled UPI #$id to ".($active?'active':'inactive'));
        jsonResponse(true, 'UPI status updated');
        break;

    case 'delete_upi':
        $id = intval($_POST['upi_id_record'] ?? 0);
        deleteRow('admin_upi_settings', 'id=?', [$id]);
        logAdmin('delete_upi', "Deleted UPI #$id");
        jsonResponse(true, 'UPI deleted');
        break;

    case 'upload_upi_qr':
        $id = intval($_POST['upi_id_record'] ?? 0);
        if ($id <= 0) jsonResponse(false, 'Invalid ID');
        if (empty($_FILES['qr_file']) || $_FILES['qr_file']['error'] !== UPLOAD_ERR_OK) {
            jsonResponse(false, 'No file uploaded');
        }
        $qrImage = handleQrUpload($_FILES['qr_file']);
        if (!$qrImage) jsonResponse(false, 'Upload failed');
        update('admin_upi_settings', ['qr_code_image'=>$qrImage], 'id=?', [$id]);
        logAdmin('upload_qr', "Uploaded QR for UPI #$id");
        jsonResponse(true, 'QR uploaded', ['image' => $qrImage]);
        break;

    case 'upload_brand_asset':
        $assetType = sanitize($_POST['asset_type'] ?? '');
        if (!in_array($assetType, ['logo','favicon'])) jsonResponse(false, 'Invalid asset type');
        if (empty($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_ERR_OK) {
            jsonResponse(false, 'No file uploaded');
        }
        $file = $_FILES['file'];
        $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        if (!in_array($ext, ['jpg','jpeg','png','webp','svg','ico'])) {
            jsonResponse(false, 'Invalid format');
        }
        if ($file['size'] > 2 * 1024 * 1024) jsonResponse(false, 'File too large (max 2MB)');
        $dir = UPLOADS_DIR . '/brand/';
        if (!is_dir($dir)) mkdir($dir, 0755, true);
        $fname = $assetType . '_' . time() . '.' . $ext;
        if (!move_uploaded_file($file['tmp_name'], $dir . $fname)) {
            jsonResponse(false, 'Upload failed');
        }
        $url = '/uploads/brand/' . $fname;
        $key = $assetType === 'logo' ? 'site_logo' : 'site_favicon';
        $existing = fetch("SELECT * FROM `site_settings` WHERE `key`=?", [$key]);
        if ($existing) {
            update('site_settings', ['value'=>$url], '`key`=?', [$key]);
        } else {
            insert('site_settings', ['key'=>$key, 'value'=>$url]);
        }
        logAdmin('upload_brand', "Uploaded $assetType: $url");
        jsonResponse(true, 'Uploaded', ['url' => $url]);
        break;

    case 'create_gift_code':
        $code = strtoupper(sanitize(trim($_POST['code'] ?? '')));
        $amount = floatval($_POST['amount'] ?? 0);
        $maxUses = intval($_POST['max_uses'] ?? 0);
        $maxPerUser = intval($_POST['max_per_user'] ?? 1);
        $minDeposit = floatval($_POST['min_deposit'] ?? 0);
        $wagering = floatval($_POST['wagering'] ?? 5);
        $expiresAt = sanitize($_POST['expires_at'] ?? '');
        if (empty($code) || $amount <= 0) jsonResponse(false, 'Code and amount required');
        $exists = fetch("SELECT * FROM `gift_codes` WHERE `code`=?", [$code]);
        if ($exists) jsonResponse(false, 'Code already exists');
        insert('gift_codes', [
            'code'=>$code,'bonus_amount'=>$amount,'max_uses'=>$maxUses,'max_per_user'=>$maxPerUser,
            'min_deposit'=>$minDeposit,'wagering_multiplier'=>$wagering,
            'expires_at'=>$expiresAt ?: null,'is_active'=>1,'created_by'=>$adminId
        ]);
        logAdmin('create_gift_code', "Created gift code $code — ₹$amount");
        jsonResponse(true, 'Gift code created');
        break;

    case 'toggle_gift_code':
        $id = intval($_POST['gift_code_id'] ?? 0);
        $active = intval($_POST['is_active'] ?? 0);
        update('gift_codes', ['is_active'=>$active], 'id=?', [$id]);
        logAdmin('toggle_gift_code', "Toggled gift code #$id");
        jsonResponse(true, 'Gift code updated');
        break;

    case 'send_notification':
        $targetType = sanitize($_POST['target'] ?? 'all');
        $targetId = intval($_POST['target_id'] ?? 0);
        $title = sanitize(trim($_POST['title'] ?? ''));
        $message = sanitize(trim($_POST['message'] ?? ''));
        $type = sanitize($_POST['type'] ?? 'system');
        if (empty($title) || empty($message)) jsonResponse(false, 'Title and message required');
        if ($targetType === 'single' && $targetId > 0) {
            addNotification($targetId, $title, $message, $type);
            logAdmin('send_notification', "Sent to user #$targetId: $title");
            jsonResponse(true, 'Notification sent');
        } elseif ($targetType === 'all') {
            $allUsers = fetchAll("SELECT `id` FROM `users` WHERE `status`='active'");
            $count = 0;
            foreach ($allUsers as $u) { addNotification($u['id'], $title, $message, $type); $count++; }
            logAdmin('send_notification', "Broadcast to $count users: $title");
            jsonResponse(true, "Sent to $count users");
        } else {
            jsonResponse(false, 'Invalid target');
        }
        break;

    default:
        jsonResponse(false, 'Invalid action');
}

function handleQrUpload(array $file): ?string
{
    $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    if (!in_array($ext, ['jpg','jpeg','png','webp'])) return null;
    if ($file['size'] > 2 * 1024 * 1024) return null;
    $dir = UPLOADS_DIR . '/upi_qr/';
    if (!is_dir($dir)) mkdir($dir, 0755, true);
    $fname = 'qr_' . time() . '_' . bin2hex(random_bytes(4)) . '.' . $ext;
    if (!move_uploaded_file($file['tmp_name'], $dir . $fname)) return null;
    return $fname;
}
