<?php
/**
 * Admin Finance API — Deposits & Withdrawals
 */
require_once __DIR__ . '/../includes/admin-check.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') jsonResponse(false, 'Invalid method');
$csrf = $_POST['csrf_token'] ?? '';
if (!verifyCSRFToken($csrf)) jsonResponse(false, 'Invalid token');

$action = sanitize($_POST['action'] ?? '');

switch ($action) {
    case 'approve_deposit':
        $depId = intval($_POST['deposit_id'] ?? 0);
        $dep = fetch("SELECT * FROM `deposits` WHERE `id`=? AND `status`='pending' LIMIT 1", [$depId]);
        if (!$dep) jsonResponse(false, 'Deposit not found or already processed');
        
        $userId = (int)$dep['user_id'];
        $amount = (float)$dep['amount'];
        $bonus = (float)$dep['bonus_amount'];
        
        try {
            $pdo = db();
            $pdo->beginTransaction();
            
            update('deposits', ['status'=>'approved','approved_by'=>$adminId,'reviewed_at'=>date('Y-m-d H:i:s')], 'id=?', [$depId]);
            addToMainWallet($userId, $amount, 'deposit', 'Deposit #'.$depId.' approved');
            
            if ($bonus > 0) {
                addToBonusWallet($userId, $bonus, 'bonus', 'First deposit bonus');
                $wagerMult = (float)getSetting('bonus_wagering_multiplier', '50');
                query("UPDATE `wallets` SET `wagering_required`=`wagering_required`+? WHERE `user_id`=?", [$bonus * $wagerMult, $userId]);
            }
            
            // Mark first deposit
            $user = getUserById($userId);
            if (empty($user['first_deposit_done']) || !(int)$user['first_deposit_done']) {
                update('users', ['first_deposit_done' => 1], 'id=?', [$userId]);
                
                // Referral registration bonus
                if (!empty($user['referred_by'])) {
                    $refBonus = (float)getSetting('referral_registration_bonus', '50');
                    if ($refBonus > 0) {
                        addToBonusWallet((int)$user['referred_by'], $refBonus, 'bonus', 'Referral bonus — user #'.$userId.' first deposit');
                        addNotification((int)$user['referred_by'], '🎁 Referral Bonus!', 'Your referral made first deposit! +₹'.number_format($refBonus), 'bonus');
                    }
                }
            }
            
            // Add wagering requirement
            $wagerMult = (float)getSetting('withdrawal_wagering_multiplier', '3');
            query("UPDATE `wallets` SET `wagering_required`=`wagering_required`+? WHERE `user_id`=?", [$amount * $wagerMult, $userId]);
            
            checkVIPUpgrade($userId);
            updateAlgorithmPhase($userId);
            addNotification($userId, '✅ Deposit Approved!', 'Your deposit of ₹'.number_format($amount).' has been credited!', 'deposit');
            
            // Update UPI daily received
            if (!empty($dep['upi_id_used'])) {
                query("UPDATE `admin_upi_settings` SET `today_received`=`today_received`+? WHERE `upi_id`=?", [$amount, $dep['upi_id_used']]);
            }
            
            $pdo->commit();
            logAdmin('approve_deposit', "Approved deposit #$depId — ₹$amount for user #$userId");
            jsonResponse(true, 'Deposit approved!');
            
        } catch (Exception $e) {
            if (db()->inTransaction()) db()->rollBack();
            jsonResponse(false, $e->getMessage());
        }
        break;
    
    case 'reject_deposit':
        $depId = intval($_POST['deposit_id'] ?? 0);
        $reason = sanitize($_POST['reason'] ?? 'Rejected by admin');
        $dep = fetch("SELECT * FROM `deposits` WHERE `id`=? AND `status`='pending'", [$depId]);
        if (!$dep) jsonResponse(false, 'Not found');
        update('deposits', ['status'=>'rejected','rejection_reason'=>$reason,'approved_by'=>$adminId,'reviewed_at'=>date('Y-m-d H:i:s')], 'id=?', [$depId]);
        addNotification((int)$dep['user_id'], '❌ Deposit Rejected', 'Deposit of ₹'.number_format((float)$dep['amount']).' rejected: '.$reason, 'deposit');
        logAdmin('reject_deposit', "Rejected deposit #$depId: $reason");
        jsonResponse(true, 'Deposit rejected');
        break;
    
    case 'approve_withdrawal':
        $wdId = intval($_POST['withdrawal_id'] ?? 0);
        $wd = fetch("SELECT * FROM `withdrawals` WHERE `id`=? AND `status`='pending'", [$wdId]);
        if (!$wd) jsonResponse(false, 'Not found');
        update('withdrawals', ['status'=>'processing','processed_by'=>$adminId,'processed_at'=>date('Y-m-d H:i:s')], 'id=?', [$wdId]);
        addNotification((int)$wd['user_id'], '🔄 Withdrawal Processing', 'Your withdrawal of ₹'.number_format((float)$wd['amount']).' is being processed.', 'withdrawal');
        logAdmin('approve_withdrawal', "Approved withdrawal #$wdId — ₹".$wd['amount']);
        jsonResponse(true, 'Withdrawal approved');
        break;
    
    case 'complete_withdrawal':
        $wdId = intval($_POST['withdrawal_id'] ?? 0);
        $wd = fetch("SELECT * FROM `withdrawals` WHERE `id`=? AND `status`='processing'", [$wdId]);
        if (!$wd) jsonResponse(false, 'Not found');
        update('withdrawals', ['status'=>'completed','processed_at'=>date('Y-m-d H:i:s')], 'id=?', [$wdId]);
        addNotification((int)$wd['user_id'], '✅ Withdrawal Complete!', 'Your withdrawal of ₹'.number_format((float)$wd['amount']).' has been sent!', 'withdrawal');
        logAdmin('complete_withdrawal', "Completed withdrawal #$wdId — ₹".$wd['amount']);
        jsonResponse(true, 'Withdrawal completed');
        break;
    
    case 'reject_withdrawal':
        $wdId = intval($_POST['withdrawal_id'] ?? 0);
        $reason = sanitize($_POST['reason'] ?? 'Rejected');
        $wd = fetch("SELECT * FROM `withdrawals` WHERE `id`=? AND `status` IN ('pending','processing')", [$wdId]);
        if (!$wd) jsonResponse(false, 'Not found');
        
        // Refund to wallet
        addToMainWallet((int)$wd['user_id'], (float)$wd['amount'], 'deposit', 'Withdrawal #'.$wdId.' rejected — refunded');
        // Reverse withdrawal stats
        query("UPDATE `users` SET `total_withdrawn`=GREATEST(0,`total_withdrawn`-?) WHERE `id`=?", [(float)$wd['amount'], (int)$wd['user_id']]);
        query("UPDATE `wallets` SET `total_withdrawn`=GREATEST(0,`total_withdrawn`-?) WHERE `user_id`=?", [(float)$wd['amount'], (int)$wd['user_id']]);
        
        update('withdrawals', ['status'=>'rejected','rejection_reason'=>$reason,'processed_by'=>$adminId,'processed_at'=>date('Y-m-d H:i:s')], 'id=?', [$wdId]);
        addNotification((int)$wd['user_id'], '❌ Withdrawal Rejected', 'Withdrawal of ₹'.number_format((float)$wd['amount']).' rejected: '.$reason.'. Amount refunded.', 'withdrawal');
        logAdmin('reject_withdrawal', "Rejected withdrawal #$wdId: $reason");
        jsonResponse(true, 'Withdrawal rejected & refunded');
        break;
    
    default:
        jsonResponse(false, 'Invalid action');
}